# Slider API Documentation

This document describes the Slider Management APIs for both admin and public endpoints.

---

## Table of Contents

1. [Admin Slider Management APIs](#admin-slider-management-apis)
   - [Get All Sliders](#get-all-sliders)
   - [Get Single Slider](#get-single-slider)
   - [Add New Slider](#add-new-slider)
   - [Update Slider](#update-slider)
   - [Delete Slider](#delete-slider)
2. [Public Slider APIs](#public-slider-apis)
   - [Get All Active Sliders](#get-all-active-sliders)
   - [Get Single Active Slider](#get-single-active-slider)

---

## Admin Slider Management APIs

All admin endpoints require authentication using a Bearer token in the Authorization header.

**Base URL:** `/api/admin/slider`

**Authentication:** Required (Admin Bearer Token)

---

### Get All Sliders

Get a list of all sliders (both active and inactive) with admin details.

**Endpoint:** `GET /api/admin/slider`

**Headers:**
```
Authorization: Bearer {admin_token}
```

**Success Response (200):**
```json
{
    "success": true,
    "data": [
        {
            "id": 1,
            "title": "Welcome to Digital Seva",
            "description": "Your one-stop solution for digital services",
            "image_url": "https://example.com/slider1.jpg",
            "link": "https://example.com/landing",
            "order": 1,
            "is_active": true,
            "created_by": 1,
            "created_at": "2025-12-06T13:46:00.000000Z",
            "updated_at": "2025-12-06T13:46:00.000000Z",
            "admin": {
                "id": 1,
                "username": "admin",
                "email": "admin@alldigtalseva.com"
            }
        },
        {
            "id": 2,
            "title": "New Services Available",
            "description": "Check out our latest digital services",
            "image_url": "https://example.com/slider2.jpg",
            "link": null,
            "order": 2,
            "is_active": false,
            "created_by": 1,
            "created_at": "2025-12-06T14:00:00.000000Z",
            "updated_at": "2025-12-06T14:00:00.000000Z",
            "admin": {
                "id": 1,
                "username": "admin",
                "email": "admin@alldigtalseva.com"
            }
        }
    ]
}
```

**Error Response (401) - Unauthorized:**
```json
{
    "success": false,
    "message": "Unauthorized. Admin access required."
}
```

**cURL Example:**
```bash
curl -X GET "https://api.alldigtalseva.com/api/admin/slider" \
  -H "Authorization: Bearer {admin_token}" \
  -H "Accept: application/json"
```

---

### Get Single Slider

Get details of a specific slider by ID.

**Endpoint:** `GET /api/admin/slider/{id}`

**URL Parameters:**
- `id` (required, integer): Slider ID

**Headers:**
```
Authorization: Bearer {admin_token}
```

**Success Response (200):**
```json
{
    "success": true,
    "data": {
        "id": 1,
        "title": "Welcome to Digital Seva",
        "description": "Your one-stop solution for digital services",
        "image_url": "https://example.com/slider1.jpg",
        "link": "https://example.com/landing",
        "order": 1,
        "is_active": true,
        "created_by": 1,
        "created_at": "2025-12-06T13:46:00.000000Z",
        "updated_at": "2025-12-06T13:46:00.000000Z",
        "admin": {
            "id": 1,
            "username": "admin",
            "email": "admin@alldigtalseva.com"
        }
    }
}
```

**Error Response (404) - Not Found:**
```json
{
    "success": false,
    "message": "Slider not found"
}
```

**cURL Example:**
```bash
curl -X GET "https://api.alldigtalseva.com/api/admin/slider/1" \
  -H "Authorization: Bearer {admin_token}" \
  -H "Accept: application/json"
```

---

### Add New Slider

Create a new slider.

**Endpoint:** `POST /api/admin/slider`

**Headers:**
```
Authorization: Bearer {admin_token}
Content-Type: application/json
```

**Request Body:**
```json
{
    "title": "Welcome to Digital Seva",
    "description": "Your one-stop solution for digital services",
    "image_url": "https://example.com/slider1.jpg",
    "link": "https://example.com/landing",
    "order": 1,
    "is_active": true
}
```

**Field Descriptions:**
- `title` (optional, string, max 255): Slider title
- `description` (optional, string): Slider description
- `image_url` (required, string, URL): URL of the slider image
- `link` (optional, string, URL): Optional link when slider is clicked
- `order` (optional, integer, min 0): Display order (default: 0)
- `is_active` (optional, boolean): Whether slider is active (default: true)

**Success Response (201):**
```json
{
    "success": true,
    "message": "Slider added successfully",
    "data": {
        "id": 1,
        "title": "Welcome to Digital Seva",
        "description": "Your one-stop solution for digital services",
        "image_url": "https://example.com/slider1.jpg",
        "link": "https://example.com/landing",
        "order": 1,
        "is_active": true,
        "created_by": 1,
        "created_at": "2025-12-06T13:46:00.000000Z",
        "updated_at": "2025-12-06T13:46:00.000000Z",
        "admin": {
            "id": 1,
            "username": "admin",
            "email": "admin@alldigtalseva.com"
        }
    }
}
```

**Error Response (422) - Validation Error:**
```json
{
    "success": false,
    "message": "Validation error",
    "errors": {
        "image_url": [
            "The image url field is required."
        ],
        "link": [
            "The link must be a valid URL."
        ]
    }
}
```

**cURL Example:**
```bash
curl -X POST "https://api.alldigtalseva.com/api/admin/slider" \
  -H "Authorization: Bearer {admin_token}" \
  -H "Content-Type: application/json" \
  -H "Accept: application/json" \
  -d '{
    "title": "Welcome to Digital Seva",
    "description": "Your one-stop solution for digital services",
    "image_url": "https://example.com/slider1.jpg",
    "link": "https://example.com/landing",
    "order": 1,
    "is_active": true
  }'
```

---

### Update Slider

Update an existing slider.

**Endpoint:** `PUT /api/admin/slider/{id}` or `PATCH /api/admin/slider/{id}`

**URL Parameters:**
- `id` (required, integer): Slider ID

**Headers:**
```
Authorization: Bearer {admin_token}
Content-Type: application/json
```

**Request Body:**
```json
{
    "title": "Updated Slider Title",
    "description": "Updated description",
    "image_url": "https://example.com/updated-slider.jpg",
    "link": "https://example.com/new-link",
    "order": 2,
    "is_active": false
}
```

**Field Descriptions:**
- All fields are optional (use `PUT` for full update, `PATCH` for partial update)
- Same validation rules as Add New Slider

**Success Response (200):**
```json
{
    "success": true,
    "message": "Slider updated successfully",
    "data": {
        "id": 1,
        "title": "Updated Slider Title",
        "description": "Updated description",
        "image_url": "https://example.com/updated-slider.jpg",
        "link": "https://example.com/new-link",
        "order": 2,
        "is_active": false,
        "created_by": 1,
        "created_at": "2025-12-06T13:46:00.000000Z",
        "updated_at": "2025-12-06T14:00:00.000000Z",
        "admin": {
            "id": 1,
            "username": "admin",
            "email": "admin@alldigtalseva.com"
        }
    }
}
```

**Error Response (404) - Not Found:**
```json
{
    "success": false,
    "message": "Slider not found"
}
```

**cURL Example:**
```bash
curl -X PUT "https://api.alldigtalseva.com/api/admin/slider/1" \
  -H "Authorization: Bearer {admin_token}" \
  -H "Content-Type: application/json" \
  -H "Accept: application/json" \
  -d '{
    "title": "Updated Slider Title",
    "is_active": false,
    "order": 2
  }'
```

---

### Delete Slider

Delete a slider.

**Endpoint:** `DELETE /api/admin/slider/{id}`

**URL Parameters:**
- `id` (required, integer): Slider ID

**Headers:**
```
Authorization: Bearer {admin_token}
```

**Success Response (200):**
```json
{
    "success": true,
    "message": "Slider deleted successfully"
}
```

**Error Response (404) - Not Found:**
```json
{
    "success": false,
    "message": "Slider not found"
}
```

**cURL Example:**
```bash
curl -X DELETE "https://api.alldigtalseva.com/api/admin/slider/1" \
  -H "Authorization: Bearer {admin_token}" \
  -H "Accept: application/json"
```

---

## Public Slider APIs

Public endpoints for fetching active sliders for the landing page. No authentication required.

**Base URL:** `/api/slider`

**Authentication:** Not required

---

### Get All Active Sliders

Get all active sliders ordered by display order. This endpoint is used for the landing page.

**Endpoint:** `GET /api/slider`

**Success Response (200):**
```json
{
    "success": true,
    "data": [
        {
            "id": 1,
            "title": "Welcome to Digital Seva",
            "description": "Your one-stop solution for digital services",
            "image_url": "https://example.com/slider1.jpg",
            "link": "https://example.com/landing",
            "order": 1
        },
        {
            "id": 3,
            "title": "New Services Available",
            "description": "Check out our latest digital services",
            "image_url": "https://example.com/slider3.jpg",
            "link": null,
            "order": 2
        }
    ]
}
```

**Note:** Only active sliders (`is_active = true`) are returned, ordered by `order` field (ascending), then by creation date (descending).

**cURL Example:**
```bash
curl -X GET "https://api.alldigtalseva.com/api/slider" \
  -H "Accept: application/json"
```

**JavaScript/Fetch Example:**
```javascript
fetch('https://api.alldigtalseva.com/api/slider')
  .then(response => response.json())
  .then(data => {
    if (data.success) {
      console.log('Sliders:', data.data);
      // Display sliders on landing page
    }
  })
  .catch(error => console.error('Error:', error));
```

---

### Get Single Active Slider

Get details of a specific active slider by ID.

**Endpoint:** `GET /api/slider/{id}`

**URL Parameters:**
- `id` (required, integer): Slider ID

**Success Response (200):**
```json
{
    "success": true,
    "data": {
        "id": 1,
        "title": "Welcome to Digital Seva",
        "description": "Your one-stop solution for digital services",
        "image_url": "https://example.com/slider1.jpg",
        "link": "https://example.com/landing",
        "order": 1
    }
}
```

**Error Response (404) - Not Found or Inactive:**
```json
{
    "success": false,
    "message": "Slider not found or inactive"
}
```

**cURL Example:**
```bash
curl -X GET "https://api.alldigtalseva.com/api/slider/1" \
  -H "Accept: application/json"
```

---

## Integration Notes

### For Landing Page Integration

1. **Fetch Sliders on Page Load:**
   ```javascript
   // Angular/React/Vue example
   async function loadSliders() {
     try {
       const response = await fetch('https://api.alldigtalseva.com/api/slider');
       const result = await response.json();
       
       if (result.success) {
         // result.data contains array of active sliders
         displaySliders(result.data);
       }
     } catch (error) {
       console.error('Failed to load sliders:', error);
     }
   }
   ```

2. **Display Order:**
   - Sliders are automatically ordered by the `order` field (ascending)
   - Lower order numbers appear first
   - If order is the same, newer sliders appear first

3. **Image URLs:**
   - Always use the `image_url` field for displaying slider images
   - Ensure images are optimized for web (recommended: WebP format, compressed)

4. **Optional Links:**
   - If `link` is provided, make the slider clickable
   - If `link` is `null`, slider is display-only

### Best Practices

1. **Image Optimization:**
   - Use high-quality images (recommended: 1920x1080 or similar aspect ratio)
   - Compress images before uploading
   - Use CDN for faster loading

2. **Order Management:**
   - Use order values like 1, 2, 3 for sequential display
   - Leave gaps (e.g., 10, 20, 30) for easy reordering

3. **Active Status:**
   - Set `is_active: false` to temporarily hide sliders without deleting
   - Useful for seasonal promotions or A/B testing

4. **Link Handling:**
   - Use absolute URLs for external links
   - Use relative paths for internal navigation
   - Always validate links before saving

---

## Error Codes

| Status Code | Description |
|------------|-------------|
| 200 | Success |
| 201 | Created successfully |
| 400 | Bad request |
| 401 | Unauthorized (Admin token required) |
| 404 | Resource not found |
| 422 | Validation error |

---

## Support

For issues or questions regarding the Slider API, please contact the development team.

