# User Signup API Documentation

Base URL: `https://api.alldigtalseva.com`

---

## User Signup

Register a new user and receive OTP via SMS.

**Endpoint:** `POST /api/user/signup`

**Request Headers:**
```
Content-Type: application/json
```

**Request Body:**
```json
{
    "email": "user@example.com",
    "phone_number": "1234567890",
    "user_type": "user"
}
```

**Parameters:**
- `email` (required, string, email): User's email address (must be unique)
- `phone_number` (required, string): User's phone number (must be unique)
- `user_type` (optional, enum): Type of user - `"user"` or `"distributor"` (default: `"user"`)

**Success Response (200):**
```json
{
    "success": true,
    "message": "OTP sent successfully to your phone number",
    "data": {
        "user_id": 1,
        "email": "user@example.com",
        "phone_number": "1234567890"
    }
}
```

**Error Response (422) - Validation Error:**
```json
{
    "success": false,
    "message": "Validation error",
    "errors": {
        "email": ["The email has already been taken."],
        "phone_number": ["The phone number has already been taken."]
    }
}
```

**Error Response (409) - User Already Verified:**
```json
{
    "success": false,
    "message": "User already exists and is verified"
}
```

**cURL Example:**
```bash
curl -X POST https://api.alldigtalseva.com/api/user/signup \
  -H "Content-Type: application/json" \
  -d '{
    "email": "user@example.com",
    "phone_number": "1234567890",
    "user_type": "user"
  }'
```

**JavaScript/Fetch Example:**
```javascript
fetch('https://api.alldigtalseva.com/api/user/signup', {
  method: 'POST',
  headers: {
    'Content-Type': 'application/json',
  },
  body: JSON.stringify({
    email: 'user@example.com',
    phone_number: '1234567890',
    user_type: 'user'
  })
})
.then(response => response.json())
.then(data => {
  console.log('Success:', data);
})
.catch((error) => {
  console.error('Error:', error);
});
```

**PHP Example:**
```php
<?php
$url = 'https://api.alldigtalseva.com/api/user/signup';
$data = [
    'email' => 'user@example.com',
    'phone_number' => '1234567890',
    'user_type' => 'user'
];

$ch = curl_init($url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Content-Type: application/json'
]);

$response = curl_exec($ch);
curl_close($ch);

$result = json_decode($response, true);
print_r($result);
?>
```

**Python Example:**
```python
import requests
import json

url = "https://api.alldigtalseva.com/api/user/signup"
payload = {
    "email": "user@example.com",
    "phone_number": "1234567890",
    "user_type": "user"
}
headers = {
    "Content-Type": "application/json"
}

response = requests.post(url, json=payload, headers=headers)
print(response.json())
```

---

## Verify OTP and Create Password

After receiving OTP via SMS, verify it and create your password to complete signup.

**Endpoint:** `POST /api/user/verify-otp`

**Request Headers:**
```
Content-Type: application/json
```

**Request Body:**
```json
{
    "email": "user@example.com",
    "otp": "123456",
    "password": "yourpassword123"
}
```

**Parameters:**
- `email` (required, string, email): User's email address used during signup
- `otp` (required, string, 6 digits): OTP received via SMS
- `password` (required, string, min: 6 characters): Password to create for the account

**Success Response (200):**
```json
{
    "success": true,
    "message": "Signup completed successfully",
    "data": {
        "id": 1,
        "email": "user@example.com",
        "phone_number": "1234567890",
        "user_type": "user",
        "wallet_balance": "0.00",
        "is_verified": true
    }
}
```

**Error Response (401) - Invalid OTP:**
```json
{
    "success": false,
    "message": "Invalid OTP"
}
```

**Error Response (401) - Expired OTP:**
```json
{
    "success": false,
    "message": "OTP has expired. Please request a new one."
}
```

**Error Response (422) - Validation Error:**
```json
{
    "success": false,
    "message": "Validation error",
    "errors": {
        "password": ["The password field is required."],
        "otp": ["The otp must be 6 characters."]
    }
}
```

**cURL Example:**
```bash
curl -X POST https://api.alldigtalseva.com/api/user/verify-otp \
  -H "Content-Type: application/json" \
  -d '{
    "email": "user@example.com",
    "otp": "123456",
    "password": "yourpassword123"
  }'
```

**JavaScript/Fetch Example:**
```javascript
fetch('https://api.alldigtalseva.com/api/user/verify-otp', {
  method: 'POST',
  headers: {
    'Content-Type': 'application/json',
  },
  body: JSON.stringify({
    email: 'user@example.com',
    otp: '123456',
    password: 'yourpassword123'
  })
})
.then(response => response.json())
.then(data => {
  console.log('Success:', data);
})
.catch((error) => {
  console.error('Error:', error);
});
```

**PHP Example:**
```php
<?php
$url = 'https://api.alldigtalseva.com/api/user/verify-otp';
$data = [
    'email' => 'user@example.com',
    'otp' => '123456',
    'password' => 'yourpassword123'
];

$ch = curl_init($url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Content-Type: application/json'
]);

$response = curl_exec($ch);
curl_close($ch);

$result = json_decode($response, true);
print_r($result);
?>
```

**Python Example:**
```python
import requests
import json

url = "https://api.alldigtalseva.com/api/user/verify-otp"
payload = {
    "email": "user@example.com",
    "otp": "123456",
    "password": "yourpassword123"
}
headers = {
    "Content-Type": "application/json"
}

response = requests.post(url, json=payload, headers=headers)
print(response.json())
```

---

## Important Notes

1. **OTP Delivery**: OTP is sent via SMS to the provided phone number using WebTextSolution API.

2. **Phone Number Format**: 
   - Phone number will be automatically formatted with country code 91 (India) if not provided
   - Example: `1234567890` becomes `911234567890`

3. **OTP Validity**: 
   - OTP is valid for 10 minutes
   - OTP is a 6-digit code

4. **User Types**:
   - `"user"` - Regular user (default)
   - `"distributor"` - Distributor user

5. **Duplicate Users**:
   - If user already exists but is not verified, a new OTP will be sent
   - If user already exists and is verified, you'll receive a 409 error

6. **Password Creation**:
   - Password is created during OTP verification (not during signup)
   - Password must be at least 6 characters long
   - Password is stored in plain text (not encrypted) as per requirements
   - You must provide password when verifying OTP

7. **Next Steps**:
   - After receiving OTP, use the `/api/user/verify-otp` endpoint with your password to complete signup
   - If OTP expires, use `/api/user/resend-otp` to get a new one

---

## Response Codes

| Status Code | Description |
|------------|-------------|
| 200 | Success - OTP sent successfully |
| 409 | Conflict - User already exists and is verified |
| 422 | Validation Error - Invalid input data |
| 500 | Server Error - Internal server error |

---

## SMS Provider Details

- **Provider**: WebTextSolution
- **Sender ID**: FARMID
- **Message Format**: "To verify your Farmer ID account, use OTP {otp} This code will expire in 10 minutes."

---

**Last Updated:** December 4, 2025

