<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Offer;
use Illuminate\Http\Request;

class OfferController extends Controller
{
    /**
     * Get All Active and Valid Offers for Public
     * Returns only offers that are active and within validity period
     */
    public function index()
    {
        $today = now()->toDateString();
        
        $offers = Offer::where('is_active', true)
            ->where('valid_from', '<=', $today)
            ->where('valid_to', '>=', $today)
            ->orderBy('order', 'asc')
            ->orderBy('created_at', 'desc')
            ->get([
                'id',
                'title',
                'description',
                'image_url',
                'offer_type',
                'discount_percentage',
                'discount_amount',
                'cashback_amount',
                'promo_code',
                'valid_from',
                'valid_to',
                'link',
                'order'
            ]);

        return response()->json([
            'success' => true,
            'data' => $offers
        ], 200);
    }

    /**
     * Get Single Active Offer (Public)
     */
    public function show($id)
    {
        $today = now()->toDateString();
        
        $offer = Offer::where('is_active', true)
            ->where('valid_from', '<=', $today)
            ->where('valid_to', '>=', $today)
            ->find($id, [
                'id',
                'title',
                'description',
                'image_url',
                'offer_type',
                'discount_percentage',
                'discount_amount',
                'cashback_amount',
                'promo_code',
                'valid_from',
                'valid_to',
                'link',
                'order'
            ]);

        if (!$offer) {
            return response()->json([
                'success' => false,
                'message' => 'Offer not found or expired'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $offer
        ], 200);
    }

    /**
     * Get Offers by Type
     * Filter offers by offer_type
     */
    public function getByType(Request $request, $type)
    {
        $validTypes = ['service_discount', 'signup_bonus', 'referral_bonus', 'cashback', 'general'];
        
        if (!in_array($type, $validTypes)) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid offer type. Valid types: ' . implode(', ', $validTypes)
            ], 400);
        }

        $today = now()->toDateString();
        
        $offers = Offer::where('is_active', true)
            ->where('offer_type', $type)
            ->where('valid_from', '<=', $today)
            ->where('valid_to', '>=', $today)
            ->orderBy('order', 'asc')
            ->orderBy('created_at', 'desc')
            ->get([
                'id',
                'title',
                'description',
                'image_url',
                'offer_type',
                'discount_percentage',
                'discount_amount',
                'cashback_amount',
                'promo_code',
                'valid_from',
                'valid_to',
                'link',
                'order'
            ]);

        return response()->json([
            'success' => true,
            'data' => $offers
        ], 200);
    }

    /**
     * Verify Promo Code
     * Check if a promo code is valid and get offer details
     */
    public function verifyPromoCode(Request $request)
    {
        $request->validate([
            'promo_code' => 'required|string'
        ]);

        $today = now()->toDateString();
        
        $offer = Offer::where('is_active', true)
            ->where('promo_code', $request->promo_code)
            ->where('valid_from', '<=', $today)
            ->where('valid_to', '>=', $today)
            ->first([
                'id',
                'title',
                'description',
                'image_url',
                'offer_type',
                'discount_percentage',
                'discount_amount',
                'cashback_amount',
                'promo_code',
                'valid_from',
                'valid_to',
                'link'
            ]);

        if (!$offer) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid or expired promo code'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'message' => 'Promo code is valid',
            'data' => $offer
        ], 200);
    }
}
