<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Blog;
use Illuminate\Http\Request;

class BlogController extends Controller
{
    /**
     * Get All Published Blogs
     * Public endpoint - returns only published blogs
     */
    public function index(Request $request)
    {
        $perPage = $request->query('per_page', 10); // Default 10 per page
        $perPage = min(max(1, $perPage), 50); // Limit between 1 and 50

        $blogs = Blog::where('is_published', true)
            ->whereNotNull('published_at')
            ->where('published_at', '<=', now())
            ->orderBy('published_at', 'desc')
            ->paginate($perPage, [
                'id',
                'title',
                'slug',
                'excerpt',
                'featured_image',
                'published_at',
                'views',
                'created_at'
            ]);

        return response()->json([
            'success' => true,
            'data' => $blogs->items(),
            'pagination' => [
                'current_page' => $blogs->currentPage(),
                'last_page' => $blogs->lastPage(),
                'per_page' => $blogs->perPage(),
                'total' => $blogs->total(),
                'from' => $blogs->firstItem(),
                'to' => $blogs->lastItem(),
            ]
        ], 200);
    }

    /**
     * Get Single Published Blog by ID
     */
    public function show($id)
    {
        $blog = Blog::where('is_published', true)
            ->whereNotNull('published_at')
            ->where('published_at', '<=', now())
            ->find($id);

        if (!$blog) {
            return response()->json([
                'success' => false,
                'message' => 'Blog not found or not published'
            ], 404);
        }

        // Increment view count
        $blog->incrementViews();

        return response()->json([
            'success' => true,
            'data' => $blog->load('admin:id,username,email')
        ], 200);
    }

    /**
     * Get Single Published Blog by Slug
     * More SEO-friendly URL
     */
    public function showBySlug($slug)
    {
        $blog = Blog::where('is_published', true)
            ->whereNotNull('published_at')
            ->where('published_at', '<=', now())
            ->where('slug', $slug)
            ->first();

        if (!$blog) {
            return response()->json([
                'success' => false,
                'message' => 'Blog not found or not published'
            ], 404);
        }

        // Increment view count
        $blog->incrementViews();

        return response()->json([
            'success' => true,
            'data' => $blog->load('admin:id,username,email')
        ], 200);
    }

    /**
     * Get Latest Published Blogs
     * Returns the most recent published blogs
     */
    public function latest(Request $request)
    {
        $limit = $request->query('limit', 5); // Default 5
        $limit = min(max(1, $limit), 20); // Limit between 1 and 20

        $blogs = Blog::where('is_published', true)
            ->whereNotNull('published_at')
            ->where('published_at', '<=', now())
            ->orderBy('published_at', 'desc')
            ->limit($limit)
            ->get([
                'id',
                'title',
                'slug',
                'excerpt',
                'featured_image',
                'published_at',
                'views',
                'created_at'
            ]);

        return response()->json([
            'success' => true,
            'data' => $blogs
        ], 200);
    }
}
