<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Admin;
use App\Models\TopUpRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Laravel\Sanctum\PersonalAccessToken;
use Carbon\Carbon;

class UserController extends Controller
{
    /**
     * Get authenticated admin from token
     */
    private function getAdmin(Request $request)
    {
        $token = $request->bearerToken();
        if (!$token) {
            return null;
        }

        $accessToken = PersonalAccessToken::findToken($token);
        if (!$accessToken) {
            return null;
        }

        $admin = $accessToken->tokenable;
        if (!$admin || !($admin instanceof Admin)) {
            return null;
        }

        return $admin;
    }

    /**
     * Get All Users
     */
    public function index(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        // Exclude CSC Users (csc_id should be null or empty)
        $query = User::where(function($q) {
            $q->whereNull('csc_id')
              ->orWhere('csc_id', '');
        });

        // Filter by user type
        if ($request->has('user_type')) {
            $query->where('user_type', $request->user_type);
        }

        // Filter by verification status
        if ($request->has('is_verified')) {
            $query->where('is_verified', $request->is_verified === 'true' || $request->is_verified === '1');
        }

        // Search by email or phone
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('email', 'like', "%{$search}%")
                  ->orWhere('phone_number', 'like', "%{$search}%")
                  ->orWhere('name', 'like', "%{$search}%");
            });
        }

        // Pagination
        $perPage = $request->input('per_page', 15);
        $perPage = min(max($perPage, 1), 100); // Limit between 1 and 100

        $users = $query->orderBy('created_at', 'desc')
            ->paginate($perPage);

        return response()->json([
            'success' => true,
            'data' => $users->items(),
            'pagination' => [
                'current_page' => $users->currentPage(),
                'per_page' => $users->perPage(),
                'total' => $users->total(),
                'last_page' => $users->lastPage(),
            ]
        ], 200);
    }

    /**
     * Get Single User
     */
    public function show(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $user = User::find($id);

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $user
        ], 200);
    }

    /**
     * Update User
     */
    public function update(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'email' => 'sometimes|nullable|email|unique:users,email,' . $id,
            'phone_number' => 'sometimes|required_without:csc_id|string|unique:users,phone_number,' . $id,
            'name' => 'nullable|string|max:255',
            'user_type' => 'nullable|in:user,distributor',
            'wallet_balance' => 'nullable|numeric|min:0',
            'is_verified' => 'nullable|boolean',
            'password' => 'nullable|string|min:6',
            'csc_id' => 'sometimes|nullable|string|unique:users,csc_id,' . $id,
            'state' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::find($id);

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found'
            ], 404);
        }

        $updateData = $request->only(['email', 'phone_number', 'name', 'user_type', 'wallet_balance', 'is_verified', 'csc_id', 'state']);
        
        // Update password if provided (plain text)
        if ($request->has('password')) {
            $updateData['password'] = $request->password;
        }

        $user->update($updateData);

        return response()->json([
            'success' => true,
            'message' => 'User updated successfully',
            'data' => $user->fresh()
        ], 200);
    }

    /**
     * Delete User
     */
    public function destroy(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $user = User::find($id);

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'User not found'
            ], 404);
        }

        $user->delete();

        return response()->json([
            'success' => true,
            'message' => 'User deleted successfully'
        ], 200);
    }

    /**
     * Get User Statistics
     */
    public function statistics(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $stats = [
            'total_users' => User::count(),
            'verified_users' => User::where('is_verified', true)->count(),
            'unverified_users' => User::where('is_verified', false)->count(),
            'regular_users' => User::where('user_type', 'user')->count(),
            'distributors' => User::where('user_type', 'distributor')->count(),
            'total_wallet_balance' => User::sum('wallet_balance'),
        ];

        return response()->json([
            'success' => true,
            'data' => $stats
        ], 200);
    }

    /**
     * Get Comprehensive Dashboard Statistics
     * Includes recharge stats, new users, top users, etc.
     */
    public function dashboardStatistics(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $today = Carbon::today();
        $thisMonth = Carbon::now()->startOfMonth();

        // Recharge Statistics
        $todayRecharges = TopUpRequest::where('status', 'approved')
            ->whereDate('approved_at', $today)
            ->get();

        $thisMonthRecharges = TopUpRequest::where('status', 'approved')
            ->where('approved_at', '>=', $thisMonth)
            ->get();

        $allRecharges = TopUpRequest::where('status', 'approved')->get();

        // User Statistics
        $todayNewUsers = User::whereDate('created_at', $today)->count();
        $thisMonthNewUsers = User::where('created_at', '>=', $thisMonth)->count();

        // Top Users by Recharge Amount
        $topUsersByAmount = TopUpRequest::where('status', 'approved')
            ->select('user_id', DB::raw('SUM(amount) as total_recharge_amount'), DB::raw('COUNT(*) as recharge_count'))
            ->groupBy('user_id')
            ->orderBy('total_recharge_amount', 'desc')
            ->limit(10)
            ->with('user:id,email,phone_number,name,user_type')
            ->get()
            ->map(function ($item) {
                return [
                    'user_id' => $item->user_id,
                    'user' => $item->user ? [
                        'id' => $item->user->id,
                        'email' => $item->user->email,
                        'phone_number' => $item->user->phone_number,
                        'name' => $item->user->name,
                        'user_type' => $item->user->user_type,
                    ] : null,
                    'total_recharge_amount' => (float) $item->total_recharge_amount,
                    'recharge_count' => $item->recharge_count,
                ];
            });

        // Top Users by Recharge Count
        $topUsersByCount = TopUpRequest::where('status', 'approved')
            ->select('user_id', DB::raw('COUNT(*) as recharge_count'), DB::raw('SUM(amount) as total_recharge_amount'))
            ->groupBy('user_id')
            ->orderBy('recharge_count', 'desc')
            ->limit(10)
            ->with('user:id,email,phone_number,name,user_type')
            ->get()
            ->map(function ($item) {
                return [
                    'user_id' => $item->user_id,
                    'user' => $item->user ? [
                        'id' => $item->user->id,
                        'email' => $item->user->email,
                        'phone_number' => $item->user->phone_number,
                        'name' => $item->user->name,
                        'user_type' => $item->user->user_type,
                    ] : null,
                    'recharge_count' => $item->recharge_count,
                    'total_recharge_amount' => (float) $item->total_recharge_amount,
                ];
            });

        $statistics = [
            'recharge_statistics' => [
                'today' => [
                    'total_amount' => (float) $todayRecharges->sum('amount'),
                    'total_count' => $todayRecharges->count(),
                    'total_credited' => (float) $todayRecharges->sum('final_credited_amount'),
                ],
                'this_month' => [
                    'total_amount' => (float) $thisMonthRecharges->sum('amount'),
                    'total_count' => $thisMonthRecharges->count(),
                    'total_credited' => (float) $thisMonthRecharges->sum('final_credited_amount'),
                ],
                'all_time' => [
                    'total_amount' => (float) $allRecharges->sum('amount'),
                    'total_count' => $allRecharges->count(),
                    'total_credited' => (float) $allRecharges->sum('final_credited_amount'),
                ],
            ],
            'user_statistics' => [
                'today_new_users' => $todayNewUsers,
                'this_month_new_users' => $thisMonthNewUsers,
                'total_users' => User::count(),
                'verified_users' => User::where('is_verified', true)->count(),
            ],
            'top_users' => [
                'by_recharge_amount' => $topUsersByAmount,
                'by_recharge_count' => $topUsersByCount,
            ],
        ];

        return response()->json([
            'success' => true,
            'data' => $statistics
        ], 200);
    }

    /**
     * Get User Recharge Details
     * Shows how much each user has recharged
     */
    public function userRechargeDetails(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $query = TopUpRequest::where('status', 'approved')
            ->select(
                'user_id',
                DB::raw('SUM(amount) as total_recharge_amount'),
                DB::raw('SUM(final_credited_amount) as total_credited_amount'),
                DB::raw('COUNT(*) as recharge_count'),
                DB::raw('MIN(approved_at) as first_recharge_date'),
                DB::raw('MAX(approved_at) as last_recharge_date')
            )
            ->groupBy('user_id')
            ->with('user:id,email,phone_number,name,user_type,wallet_balance');

        // Filter by date range
        if ($request->has('start_date')) {
            $query->whereDate('approved_at', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $query->whereDate('approved_at', '<=', $request->end_date);
        }

        // Filter by user type
        if ($request->has('user_type')) {
            $query->whereHas('user', function($q) use ($request) {
                $q->where('user_type', $request->user_type);
            });
        }

        // Sort options
        $sortBy = $request->input('sort_by', 'total_recharge_amount'); // total_recharge_amount, recharge_count, last_recharge_date
        $sortOrder = $request->input('sort_order', 'desc'); // asc, desc

        if ($sortBy === 'recharge_count') {
            $query->orderBy('recharge_count', $sortOrder);
        } elseif ($sortBy === 'last_recharge_date') {
            $query->orderBy('last_recharge_date', $sortOrder);
        } else {
            $query->orderBy('total_recharge_amount', $sortOrder);
        }

        // Pagination
        $perPage = $request->input('per_page', 15);
        $perPage = min(max($perPage, 1), 100);

        $results = $query->paginate($perPage);

        $data = $results->map(function ($item) {
            return [
                'user_id' => $item->user_id,
                'user' => $item->user ? [
                    'id' => $item->user->id,
                    'email' => $item->user->email,
                    'phone_number' => $item->user->phone_number,
                    'name' => $item->user->name,
                    'user_type' => $item->user->user_type,
                    'wallet_balance' => (float) $item->user->wallet_balance,
                ] : null,
                'total_recharge_amount' => (float) $item->total_recharge_amount,
                'total_credited_amount' => (float) $item->total_credited_amount,
                'recharge_count' => $item->recharge_count,
                'first_recharge_date' => $item->first_recharge_date,
                'last_recharge_date' => $item->last_recharge_date,
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $data,
            'pagination' => [
                'current_page' => $results->currentPage(),
                'per_page' => $results->perPage(),
                'total' => $results->total(),
                'last_page' => $results->lastPage(),
            ]
        ], 200);
    }

    /**
     * Get Top Users by Recharge
     */
    public function topUsersByRecharge(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $sortBy = $request->input('sort_by', 'amount'); // amount, count
        $limit = min(max($request->input('limit', 10), 1), 50);

        $query = TopUpRequest::where('status', 'approved')
            ->select(
                'user_id',
                DB::raw('SUM(amount) as total_recharge_amount'),
                DB::raw('COUNT(*) as recharge_count')
            )
            ->groupBy('user_id')
            ->with('user:id,email,phone_number,name,user_type');

        // Filter by date range
        if ($request->has('start_date')) {
            $query->whereDate('approved_at', '>=', $request->start_date);
        }

        if ($request->has('end_date')) {
            $query->whereDate('approved_at', '<=', $request->end_date);
        }

        if ($sortBy === 'count') {
            $query->orderBy('recharge_count', 'desc');
        } else {
            $query->orderBy('total_recharge_amount', 'desc');
        }

        $results = $query->limit($limit)->get();

        $data = $results->map(function ($item) {
            return [
                'user_id' => $item->user_id,
                'user' => $item->user ? [
                    'id' => $item->user->id,
                    'email' => $item->user->email,
                    'phone_number' => $item->user->phone_number,
                    'name' => $item->user->name,
                    'user_type' => $item->user->user_type,
                ] : null,
                'total_recharge_amount' => (float) $item->total_recharge_amount,
                'recharge_count' => $item->recharge_count,
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $data,
            'sort_by' => $sortBy,
            'limit' => $limit
        ], 200);
    }
        /**
     * Get CSC Users
     * Lists users with CSC ID, including their password and state
     */
    public function getCscUsers(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $query = User::whereNotNull('csc_id')
            ->where('csc_id', '!=', '')
            ->select('id', 'name', 'email', 'phone_number', 'csc_id', 'state', 'password', 'wallet_balance', 'is_verified', 'created_at');

        // Search functionality
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('csc_id', 'like', "%{$search}%")
                  ->orWhere('phone_number', 'like', "%{$search}%")
                  ->orWhere('state', 'like', "%{$search}%");
            });
        }

        // Filter by State
        if ($request->has('state')) {
            $query->where('state', $request->state);
        }

        $perPage = $request->input('per_page', 15);
        $users = $query->orderBy('created_at', 'desc')->paginate($perPage);

        // Make visible password for these specific users in this API response only
        $users->getCollection()->each(function ($user) {
            $user->makeVisible(['password']);
        });

        return response()->json([
            'success' => true,
            'data' => $users->items(),
            'pagination' => [
                'current_page' => $users->currentPage(),
                'per_page' => $users->perPage(),
                'total' => $users->total(),
                'last_page' => $users->lastPage(),
            ]
        ], 200);
    }
}
