<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\Upi;
use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Laravel\Sanctum\PersonalAccessToken;

class UpiController extends Controller
{
    /**
     * Get authenticated admin from token
     */
    private function getAdmin(Request $request)
    {
        $token = $request->bearerToken();
        if (!$token) {
            return null;
        }

        $accessToken = PersonalAccessToken::findToken($token);
        if (!$accessToken) {
            return null;
        }

        $admin = $accessToken->tokenable;
        if (!$admin || !($admin instanceof Admin)) {
            return null;
        }

        return $admin;
    }

    /**
     * Add New UPI
     */
    public function store(Request $request)
    {
        // Check admin authentication
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'upi_id' => 'required|string|unique:upis,upi_id',
            'name' => 'nullable|string|max:255',
            'status' => 'nullable|in:active,inactive',
            'is_default' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        // If setting as default, unset other default UPIs
        if ($request->is_default) {
            Upi::where('is_default', true)->update(['is_default' => false]);
        }

        $upi = Upi::create([
            'upi_id' => $request->upi_id,
            'name' => $request->name,
            'status' => $request->status ?? 'active',
            'is_default' => $request->is_default ?? false,
            'created_by' => $admin->id,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'UPI added successfully',
            'data' => $upi
        ], 201);
    }

    /**
     * Update UPI
     */
    public function update(Request $request, $id)
    {
        // Check admin authentication
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'upi_id' => 'sometimes|required|string|unique:upis,upi_id,' . $id,
            'name' => 'nullable|string|max:255',
            'status' => 'nullable|in:active,inactive',
            'is_default' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $upi = Upi::find($id);

        if (!$upi) {
            return response()->json([
                'success' => false,
                'message' => 'UPI not found'
            ], 404);
        }

        // If setting as default, unset other default UPIs
        if ($request->has('is_default') && $request->is_default) {
            Upi::where('is_default', true)->where('id', '!=', $id)->update(['is_default' => false]);
        }

        $upi->update($request->only(['upi_id', 'name', 'status', 'is_default']));

        return response()->json([
            'success' => true,
            'message' => 'UPI updated successfully',
            'data' => $upi->fresh()
        ], 200);
    }

    /**
     * Get All UPIs
     */
    public function index(Request $request)
    {
        // Check admin authentication
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $upis = Upi::orderBy('is_default', 'desc')
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $upis
        ], 200);
    }

    /**
     * Get Single UPI
     */
    public function show(Request $request, $id)
    {
        // Check admin authentication
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $upi = Upi::find($id);

        if (!$upi) {
            return response()->json([
                'success' => false,
                'message' => 'UPI not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $upi
        ], 200);
    }

    /**
     * Delete UPI
     */
    public function destroy(Request $request, $id)
    {
        // Check admin authentication
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $upi = Upi::find($id);

        if (!$upi) {
            return response()->json([
                'success' => false,
                'message' => 'UPI not found'
            ], 404);
        }

        $upi->delete();

        return response()->json([
            'success' => true,
            'message' => 'UPI deleted successfully'
        ], 200);
    }
}
