<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\TrendingOffer;
use App\Models\Admin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Laravel\Sanctum\PersonalAccessToken;

class TrendingOfferController extends Controller
{
    /**
     * Get authenticated admin from token
     */
    private function getAdmin(Request $request)
    {
        $token = $request->bearerToken();
        if (!$token) {
            return null;
        }

        $accessToken = PersonalAccessToken::findToken($token);
        if (!$accessToken) {
            return null;
        }

        $admin = $accessToken->tokenable;
        if (!$admin || !($admin instanceof Admin)) {
            return null;
        }

        return $admin;
    }

    /**
     * Get All Trending Offers
     */
    public function index(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $trendingOffers = TrendingOffer::with('admin')
            ->orderBy('order', 'asc')
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $trendingOffers
        ], 200);
    }

    /**
     * Get Single Trending Offer
     */
    public function show(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $trendingOffer = TrendingOffer::with('admin')->find($id);

        if (!$trendingOffer) {
            return response()->json([
                'success' => false,
                'message' => 'Trending offer not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $trendingOffer
        ], 200);
    }

    /**
     * Add New Trending Offer
     */
    public function store(Request $request)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'text' => 'required|string',
            'order' => 'nullable|integer|min:0',
            'is_active' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $trendingOffer = TrendingOffer::create([
            'text' => $request->text,
            'order' => $request->order ?? 0,
            'is_active' => $request->is_active ?? true,
            'created_by' => $admin->id,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Trending offer added successfully',
            'data' => $trendingOffer->load('admin')
        ], 201);
    }

    /**
     * Update Trending Offer
     */
    public function update(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'text' => 'sometimes|required|string',
            'order' => 'nullable|integer|min:0',
            'is_active' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $trendingOffer = TrendingOffer::find($id);

        if (!$trendingOffer) {
            return response()->json([
                'success' => false,
                'message' => 'Trending offer not found'
            ], 404);
        }

        $trendingOffer->update($request->only(['text', 'order', 'is_active']));

        return response()->json([
            'success' => true,
            'message' => 'Trending offer updated successfully',
            'data' => $trendingOffer->fresh()->load('admin')
        ], 200);
    }

    /**
     * Delete Trending Offer
     */
    public function destroy(Request $request, $id)
    {
        $admin = $this->getAdmin($request);
        if (!$admin) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. Admin access required.'
            ], 401);
        }

        $trendingOffer = TrendingOffer::find($id);

        if (!$trendingOffer) {
            return response()->json([
                'success' => false,
                'message' => 'Trending offer not found'
            ], 404);
        }

        $trendingOffer->delete();

        return response()->json([
            'success' => true,
            'message' => 'Trending offer deleted successfully'
        ], 200);
    }
}
